<?php

use Alley\WP\Block_Converter\Block_Converter;
use Hostinger\WpHelper\Config;
use Hostinger\WpHelper\Utils as Helper;
use Hostinger\WpHelper\Constants;
use Hostinger\WpHelper\Requests\Client;
use Hostinger\Amplitude\AmplitudeManager;

class Hostinger_Ai_Assistant_Content_Generation {

    private Hostinger_Ai_Assistant_Errors $error_handler;
    private Hostinger_Ai_Assistant_Helper $helper;
    private Hostinger_Ai_Assistant_Amplitude $amplitude;

    private const ALLOWED_LENGTHS = [
        'default' => [
            'short'  => [ 'min' => 150, 'max' => 300 ],
            'medium' => [ 'min' => 600, 'max' => 1200 ],
            'long'   => [ 'min' => 1500, 'max' => 2600 ],
        ],
        'product' => [
            'short'  => [ 'min' => 75, 'max' => 150 ],
            'medium' => [ 'min' => 300, 'max' => 600 ],
            'long'   => [ 'min' => 750, 'max' => 1300 ],
        ],
    ];

    function __construct() {
        $this->error_handler = new Hostinger_Ai_Assistant_Errors();
        $this->helper        = new Hostinger_Ai_Assistant_Helper();
        $helper              = new Helper();
        $config              = new Config();
        $client              = new Client( $config->getConfigValue( 'base_rest_uri', Constants::HOSTINGER_REST_URI ), [
                Config::TOKEN_HEADER  => $helper->getApiToken(),
                Config::DOMAIN_HEADER => $helper->getHostInfo(),
            ] );

        $amplitudeManager = new AmplitudeManager( $helper, $config, $client );
        $this->amplitude  = new Hostinger_Ai_Assistant_Amplitude( $amplitudeManager );
    }

    public function get_or_create_category( string $category ): int {
        $category_id = category_exists( $category );
        if ( ! $category_id ) {
            $category_data = array(
                'cat_name'             => $category,
                'category_description' => '',
                'category_nicename'    => sanitize_title( $category ),
            );
            $category_id   = wp_insert_category( $category_data );
        }

        return $category_id;
    }

    public function get_or_create_tags( string $tags ): array {
        $tags_array = explode( ',', $tags );
        $tags_ids   = array();
        foreach ( $tags_array as $tag_name ) {
            $tag = term_exists( $tag_name, 'post_tag' );
            if ( ! $tag ) {
                $tag_data = array(
                    'slug' => sanitize_title( $tag_name ),
                    'name' => $tag_name,
                );
                $tag_id   = wp_insert_term( $tag_data['name'], 'post_tag', $tag_data );
                if ( is_wp_error( $tag_id ) ) {
                    error_log( 'Failed tag create ' . print_r( $tag_id, true ) );
                }
                $tag_id = $tag_id['term_id'];
            } else {
                $tag_id = $tag['term_id'];
            }
            $tags_ids[] = (int)$tag_id;
        }

        return $tags_ids;
    }

    public function create_post( string $post_type, string $title, string $content, int $category_id, array $tags_ids, string $correlation_id, string $post_status = 'draft' ): int {
        $post_data = array(
            'post_title'    => $title,
            'post_content'  => $content,
            'post_status'   => $post_status,
            'post_category' => array( $category_id ),
            'tags_input'    => $tags_ids,
            'post_type'     => $post_type,
        );

        $post_id = wp_insert_post( $post_data );

        update_post_meta( $post_id, 'hts_correlation_id', $correlation_id );

        // Save meta field when content generated by AI.
        update_post_meta( $post_id, 'hostinger_ai_generated', true );

        $this->amplitude->ai_content_saved( $post_type, $post_id, 'ai_assistant_ui', [ 'correlation_id' => $correlation_id ] );

        if ( $post_status === 'publish' ) {
            $correlation_id = get_post_meta( $post_id, 'hts_correlation_id', true );
            $this->amplitude->ai_content_published( $post_type, $post_id, 'ai_assistant_ui', [ 'correlation_id' => $correlation_id ] );
            delete_post_meta($post_id, 'hts_correlation_id');
        }

        return $post_id;
    }

    public function get_public_post_types(): array {
        $post_types                        = get_post_types( array(
            'public' => true,
        ) );
        $post_types_with_content_and_title = array();

        foreach ( $post_types as $post_type ) {
            if ( post_type_supports( $post_type, 'editor' ) && post_type_supports( $post_type, 'title' ) ) {
                $post_types_with_content_and_title[] = $post_type;
            }
        }

        return $post_types_with_content_and_title;
    }

    public function process_post_action( string $action ): void {
        $error_msg    = $this->error_handler->get_error_message( 'action_failed' );
        $server_error = $this->error_handler->get_error_message( 'server_error' );

        try {
            // Extract and validate input data
            $post_data = $this->get_post_data( $action );

            // Process the post content with images if needed
            $post_data = $this->process_post_content_with_images( $post_data );

            // Create the post
            $post_id = $this->create_post_with_metadata( $post_data, $action );

            // Send response to the client
            $this->send_response( $post_id, $action, $error_msg );
        } catch ( Exception $exception ) {
            $this->helper->ajax_error_message( 'Error: ' . $exception->getMessage(), $server_error );
        }
    }

    /**
     * Get and validate post data from the request
     */
    private function get_post_data( string $action ): array {
        $nonce           = isset( $_POST['nonce'] ) ? sanitize_text_field( $_POST['nonce'] ) : '';
        $correlation_id  = isset( $_POST['correlation_id'] ) ? sanitize_text_field( $_POST['correlation_id'] ) : '';
        $title           = isset( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
        $thumbnail_image = isset( $_POST['thumbnail_image'] ) ? filter_var( $_POST['thumbnail_image'], FILTER_VALIDATE_BOOLEAN ) : false;
        $content_image   = isset( $_POST['content_image'] ) ? filter_var( $_POST['content_image'], FILTER_VALIDATE_BOOLEAN ) : false;
        $content         = isset( $_POST['content'] ) ? wp_kses_post( $this->helper->sanitize_html_string( $_POST['content'] ) ) : '';
        $image_url       = isset( $_POST['image_url'] ) ? sanitize_text_field( $_POST['image_url'] ) : '';
        $category        = isset( $_POST['category'] ) ? sanitize_text_field( $_POST['category'] ) : '';
        $tags            = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';
        $post_type       = isset( $_POST['post_type'] ) ? sanitize_text_field( $_POST['post_type'] ) : '';
        $seo_keywords    = isset( $_POST['seo_keywords'] ) ? sanitize_text_field( $_POST['seo_keywords'] ) : '';
        $seo_description = isset( $_POST['seo_description'] ) ? sanitize_text_field( $_POST['seo_description'] ) : '';
        $post_type       = $this->validate_post_type( $post_type );

        // Verify nonce
        if ( ! wp_verify_nonce( $nonce, ( $action === 'create' ) ? 'create_post' : 'publish_post' ) ) {
            $error_msg = $this->error_handler->get_error_message( 'action_failed' );
            $this->helper->ajax_error_message( $error_msg, $error_msg );
            exit;
        }

        return [
            'title'           => $title,
            'content'         => $content,
            'post_type'       => $post_type,
            'category'        => $category,
            'tags'            => $tags,
            'thumbnail_image' => $thumbnail_image,
            'content_image'   => $content_image,
            'image_url'       => $image_url,
            'seo_keywords'    => $seo_keywords,
            'seo_description' => $seo_description,
            'correlation_id'  => $correlation_id,
        ];
    }

    /**
     * Process taxonomy data - categories and tags
     */
    private function process_taxonomies( array $post_data ): array {
        $category_id = 0;
        $tags_ids    = [];

        if ( $this->helper->has_taxonomy_for_post_type( $post_data['post_type'], 'category' ) ) {
            $category_id = $this->get_or_create_category( $post_data['category'] );
        }

        if ( $this->helper->has_taxonomy_for_post_type( $post_data['post_type'], 'post_tag' ) ) {
            $tags_ids = $this->get_or_create_tags( $post_data['tags'] );
        }

        $post_data['category_id'] = $category_id;
        $post_data['tags_ids']    = $tags_ids;

        return $post_data;
    }

    /**
     * Process post content with images if needed
     */
    private function process_post_content_with_images( array $post_data ): array {
        $image                       = '';
        $external_featured_image_url = '';

        // Process taxonomy data
        $post_data = $this->process_taxonomies( $post_data );

        // Handle featured and content images
        if ( $this->helper->post_type_supports_featured_image( $post_data['post_type'] )
             && ( $post_data['thumbnail_image'] || $post_data['content_image'] ) ) {
            if ( ! empty( $post_data['image_url'] ) ) {
                $external_featured_image_url = $post_data['image_url'];

                // If content image is requested, prepare image based on editor type
                if ( $post_data['content_image'] ) {
                    $image = $this->format_content_image( $external_featured_image_url, $post_data['title'], $post_data['post_type'] );
                }
            }
        }

        // Add image to content if needed
        if ( ! empty( $image ) && $post_data['content_image'] ) {
            $post_data['content'] = $this->add_image_to_content( $image, $post_data['content'], $post_data['post_type'] );
        }

        $post_data['external_featured_image_url'] = $external_featured_image_url;

        return $post_data;
    }

    /**
     * Format content image based on editor type
     */
    private function format_content_image( string $image_url, string $title, string $post_type ): string {
        if ( use_block_editor_for_post_type( $post_type ) ) {
            // For Gutenberg/Block Editor sites
            return sprintf( '<!-- wp:image {"sizeSlug":"large","className":"hts-content-image"} -->
<figure class="wp-block-image size-large hts-content-image"><img src="%s" alt="%s"/></figure>
<!-- /wp:image -->', esc_url( $image_url ), esc_attr( $title ) );
        } else {
            // For Classic Editor sites
            return sprintf( '<p><img src="%s" alt="%s" class="hts-content-image size-large wp-image-0" /></p>', esc_url( $image_url ), esc_attr( $title ) );
        }
    }

    /**
     * Add image to content
     */
    private function add_image_to_content( string $image, string $content, string $post_type ): string {
        // Add the image to the beginning of the content
        $content = $image . $content;

        // Skip Block_Converter for non-block editor sites or if we've already formatted the content as blocks
        if ( ! use_block_editor_for_post_type( $post_type ) || strpos( $content, '<!-- wp:' ) === 0 ) {
            // Don't convert to blocks, use content as is
            return $content;
        } else {
            // For Gutenberg sites where content isn't already in block format
            $converter = new Block_Converter( $content );

            return $converter->convert();
        }
    }

    /**
     * Create the post with all metadata
     */
    private function create_post_with_metadata( array $post_data, string $action ): int {
        $seo = new Hostinger_Ai_Assistant_Seo();

        // Create the post
        $post_status = ( $action === 'create' ) ? '' : 'publish';
        $post_id     = $this->create_post(
            $post_data['post_type'],
            $post_data['title'],
            $post_data['content'],
 $post_data['category_id'] ?? 0,
    $post_data['tags_ids'] ?? [],
 $post_data['correlation_id'] ?? '',
            $post_status );

        // Add the external featured image URL if available and thumbnail is requested
        if ( ! empty( $post_data['external_featured_image_url'] ) && $post_data['thumbnail_image'] ) {
            update_post_meta( $post_id, '_thumbnail_ext_url', $post_data['external_featured_image_url'] );
            update_post_meta( $post_id, '_thumbnail_is_external', true );
        }

        // Add SEO metadata
        $seo->add_seo_meta_data( $post_data['seo_keywords'], $post_data['seo_description'], $post_id );

        return $post_id;
    }

    /**
     * Send response back to the client
     */
    private function send_response( int $post_id, string $action, string $error_msg ): void {
        if ( $post_id ) {
            if ( $action === 'create' ) {
                $editor_url = 'post.php?action=edit&post=' . $post_id;
                wp_send_json_success( $editor_url );
            } else {
                $post_link = get_permalink( $post_id );
                wp_send_json_success( $post_link );
            }
        } else {
            wp_send_json_error( $error_msg );
        }
    }

    public function validate_post_type( $post_type ): string {
        $allowed_post_types = array( 'post', 'product', 'page' );

        return in_array( $post_type, $allowed_post_types ) ? $post_type : 'page';
    }

    public function map_post_type( string $post_type ): string {
        switch ( $post_type ) {
            case 'post':
                return 'blog_post';
            case 'product':
                return 'product_description';
            default:
                return $post_type;
        }
    }

    public function validate_content_length( $content_length, $is_product = false ): string {
        $allowed_lengths = $is_product ? self::ALLOWED_LENGTHS['product'] : self::ALLOWED_LENGTHS['default'];

        if ( ! isset( $allowed_lengths[$content_length] ) ) {
            $content_length = 'short';
        }

        $length_range = $allowed_lengths[$content_length];

        return $length_range['min'] . '-' . $length_range['max'];
    }

}
